/**
* \file: PointerListener.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Digital iPod Out - Wayland Adapter
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <linux/input.h>
#include <adit_logging.h>
#include <dipo_plugin.h>
#include "PointerListener.h"

LOG_IMPORT_CONTEXT(dipo_wl);

namespace adit { namespace carplay
{

using namespace std;

struct wl_pointer_listener PointerListener::Listener =
{
    PointerListener::handleEnter,
    PointerListener::handleLeave,
    PointerListener::handleMotion,
    PointerListener::handleButton,
    PointerListener::handleAxis,
    PointerListener::handleFrame,
    PointerListener::handleAxis_Source,
    PointerListener::handleAxis_Stop,
    PointerListener::handleAxis_Discrete
};

PointerListener::PointerListener(IInputReceiver& inReceiver, HIDDigitizer& inDigitizer, int inWidth,
        int inHeight, bool inVerbose) :
        receiver(inReceiver), hid(inDigitizer)
{
    x = 0;
    y = 0;
    buttons = 0;

    width = inWidth;
    height = inHeight;
    verbose = inVerbose;
}

PointerListener::~PointerListener()
{
    // do nothing
}

void PointerListener::handleEnter(void* inData, struct wl_pointer* inPointer, uint32_t inSerial,
        struct wl_surface* inSurface, wl_fixed_t inX, wl_fixed_t inY)
{
    (void)inPointer;
    (void)inSerial;
    (void)inSurface;

    auto me = static_cast<PointerListener*>(inData);

    me->x = (float)wl_fixed_to_double(inX) / me->height;
    me->y = (float)wl_fixed_to_double(inY) / me->width;
}

void PointerListener::handleLeave(void* inData, struct wl_pointer* inPointer, uint32_t inSerial,
        struct wl_surface* inSurface)
{
    (void)inData;
    (void)inPointer;
    (void)inSerial;
    (void)inSurface;

    // do nothing
}

void PointerListener::handleMotion(void* inData, struct wl_pointer* inPointer, uint32_t inTime,
                    wl_fixed_t inX, wl_fixed_t inY)
{
    (void)inPointer;
    (void)inTime;

    auto me = static_cast<PointerListener*>(inData);

    me->x = (float)wl_fixed_to_double(inX) / me->width;
    me->y = (float)wl_fixed_to_double(inY) / me->height;

    if (me->buttons != 0)
    {
        // motion in pressed state
        HIDInputReportWithData report;
        if (me->hid.Down(me->x, me->y, 0, report))
            me->receiver.SendInput(report.report);
    }
}

void PointerListener::handleButton(void* inData, struct wl_pointer* inPointer, uint32_t inSerial,
                    uint32_t inTime, uint32_t inButton, uint32_t inState)
{
    (void)inPointer;
    (void)inSerial;
    (void)inTime;

    auto me = static_cast<PointerListener*>(inData);

    uint8_t button = 0;
    switch (inButton)
    {
    case BTN_TOUCH:
        button |= 1;
        break;
    case BTN_LEFT:
        button |= 1 << 1;
        break;
    case BTN_RIGHT:
        button |= 1 << 2;
        break;
    case BTN_MIDDLE:
        button |= 1 << 3;
        break;
    }

    bool previouslyPressed = (me->buttons != 0);

    if (inState == 0)
    {
        // remove all up bits
        me->buttons &= ~button;
    }
    else
    {
        // add down bits
        me->buttons |= button;
    }

    if (!previouslyPressed && me->buttons != 0)
    {
        // first button was pressed
        if (me->verbose)
        {
            LOGD_VERBOSE((dipo_wl, "PointerListener button down at %.3fx%.3f", me->x, me->y));
        }
        HIDInputReportWithData report;
        if (me->hid.Down(me->x, me->y, 0, report))
            me->receiver.SendInput(report.report);
    }
    else if (previouslyPressed && me->buttons == 0)
    {
        // last button was released
        if (me->verbose)
        {
            LOGD_VERBOSE((dipo_wl, "PointerListener button up at %.3fx%.3f", me->x, me->y));
        }
        HIDInputReportWithData report;
        if (me->hid.Up(0, report))
            me->receiver.SendInput(report.report);
    }
    else
    {
        // no change, no SendInput
    }
}

void PointerListener::handleAxis(void* inData, struct wl_pointer* inPointer, uint32_t inTime,
        uint32_t inAxis, wl_fixed_t inValue)
{
    (void)inData;
    (void)inPointer;
    (void)inTime;
    (void)inAxis;
    (void)inValue;
}

void PointerListener::handleFrame(void* inData, struct wl_pointer* inPointer)
{
    (void)inData;
	(void)inPointer;
}

void PointerListener::handleAxis_Source(void* inData, struct wl_pointer* inPointer, uint32_t inAxis_source)
{
    (void)inData;
    (void)inPointer;
    (void)inAxis_source;
}

void PointerListener::handleAxis_Stop(void* inData, struct wl_pointer* inPointer, uint32_t inTime, uint32_t inAxis)
{
    (void)inData;
    (void)inPointer;
    (void)inTime;
    (void)inAxis;
}

void PointerListener::handleAxis_Discrete(void* inData, struct wl_pointer* inPointer, uint32_t inAxis, int32_t inDiscrete)
{
    (void)inData;
    (void)inPointer;
    (void)inAxis;
    (void)inDiscrete;
}
} } // namespace adit { namespace carplay
